import AppLayout from '@/layouts/app-layout';
import { type BreadcrumbItem } from '@/types';
import { Head } from '@inertiajs/react';
import { BarElement, CategoryScale, Chart as ChartJS, Legend, LinearScale, Title, Tooltip } from 'chart.js';
import { useEffect, useState } from 'react';
import { Bar } from 'react-chartjs-2';

ChartJS.register(CategoryScale, LinearScale, BarElement, Title, Tooltip, Legend);

const breadcrumbs: BreadcrumbItem[] = [{ title: 'Dashboard', href: '/dashboard' }];

interface DashboardProps {
    totalAnggota: number;
    anggotaHadir: number;
    anggotaTidakHadir: number;
    chartData: Array<{ date: string; value: number }>;
}

export default function Dashboard({ totalAnggota, anggotaHadir, anggotaTidakHadir, chartData }: DashboardProps) {
    const chartLabels = chartData.map((item) => item.date);
    const chartValues = chartData.map((item) => item.value);

    // UseEffect for rerendering chart data when new data is passed
    const [chartConfig, setChartConfig] = useState({
        labels: chartLabels,
        datasets: [
            {
                label: 'Jumlah Kehadiran',
                data: chartValues,
                backgroundColor: 'rgba(54, 162, 235, 0.6)',
                borderColor: 'rgba(54, 162, 235, 1)',
                borderWidth: 1,
            },
        ],
    });

    useEffect(() => {
        setChartConfig({
            labels: chartLabels,
            datasets: [
                {
                    label: 'Jumlah Kehadiran',
                    data: chartValues,
                    backgroundColor: 'rgba(54, 162, 235, 0.6)',
                    borderColor: 'rgba(54, 162, 235, 1)',
                    borderWidth: 1,
                },
            ],
        });
    }, [chartData]);

    const options = {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            title: {
                display: true,
                text: 'Statistik Kehadiran Anggota',
                font: { size: 20, weight: 'bold' },
                color: '#333',
            },
            tooltip: {
                backgroundColor: 'rgba(0,0,0,0.7)',
                titleColor: '#fff',
                bodyColor: '#fff',
                borderColor: '#75b9b2',
                borderWidth: 1,
                padding: 10,
            },
            legend: {
                position: 'top',
                labels: {
                    boxWidth: 20,
                    padding: 10,
                    font: { size: 14 },
                },
            },
        },
        scales: {
            x: {
                title: { display: true, text: 'Tanggal', font: { size: 14 } },
                grid: { display: true, color: 'rgba(200,200,200,0.3)' },
            },
            y: {
                title: { display: true, text: 'Jumlah Hadir', font: { size: 14 } },
                grid: { display: true, color: 'rgba(200,200,200,0.3)' },
            },
        },
    };

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Dashboard" />
            <div className="flex h-full flex-col gap-8 rounded-xl p-8">
                <div className="grid gap-6 md:grid-cols-3">
                    {/* Total Anggota */}
                    <div className="relative aspect-video overflow-hidden rounded-xl border bg-white shadow-lg dark:border-gray-600 dark:bg-gray-800">
                        <div className="flex flex-col items-center justify-center p-6 text-center">
                            <h3 className="text-xl font-semibold text-gray-700 dark:text-white">Total Anggota</h3>
                            <p className="text-3xl font-bold text-indigo-600 dark:text-indigo-400">{totalAnggota}</p>
                        </div>
                    </div>

                    {/* Anggota Hadir */}
                    <div className="relative aspect-video overflow-hidden rounded-xl border bg-white shadow-lg dark:border-gray-600 dark:bg-gray-800">
                        <div className="flex flex-col items-center justify-center p-6 text-center">
                            <h3 className="text-xl font-semibold text-gray-700 dark:text-white">Anggota Hadir</h3>
                            <p className="text-3xl font-bold text-green-600 dark:text-green-400">{anggotaHadir}</p>
                        </div>
                    </div>

                    {/* Anggota Tidak Hadir */}
                    <div className="relative aspect-video overflow-hidden rounded-xl border bg-white shadow-lg dark:border-gray-600 dark:bg-gray-800">
                        <div className="flex flex-col items-center justify-center p-6 text-center">
                            <h3 className="text-xl font-semibold text-gray-700 dark:text-white">Anggota Tidak Hadir</h3>
                            <p className="text-3xl font-bold text-red-600 dark:text-red-400">{anggotaTidakHadir}</p>
                        </div>
                    </div>
                </div>

                {/* Grafik */}
                <div className="relative min-h-[50vh] overflow-hidden rounded-xl border bg-white shadow-lg dark:border-gray-600 dark:bg-gray-800">
                    <div className="p-8">
                        <h2 className="text-2xl font-semibold text-gray-800 dark:text-white">Statistik Kehadiran Anggota</h2>
                        <div className="mt-6">
                            <Bar data={chartConfig} options={options} />
                        </div>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
