<?php

namespace App\Http\Controllers;

use App\Models\Anggota;
use App\Models\Kehadiran;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class KehadiranController extends Controller
{
    public function index()
    {
        return Inertia::render('ScanKehadiran');
    }

    public function store(Request $request)
    {
        $request->validate([
            'qr_code_data' => 'required|string|uuid',
        ]);

        $qrCodeData = $request->input('qr_code_data');

        $anggota = Anggota::where('uuid', $qrCodeData)->first();

        if (!$anggota) {
            Log::warning('Anggota tidak ditemukan untuk UUID: ' . $qrCodeData);
            return response()->json([
                'status' => 'error',
                'message' => 'Anggota tidak ditemukan.'
            ], 404);
        }

        $today = now()->toDateString();
        $hasAttendedToday = Kehadiran::where('anggota_id', $anggota->id)
            ->whereDate('waktu_hadir', $today)
            ->exists();

        if ($hasAttendedToday) {
            Log::info('Percobaan kehadiran ganda untuk Anggota ID: ' . $anggota->id . ' pada ' . $today);
            return response()->json([
                'status' => 'error',
                'message' => 'Anda sudah mencatat kehadiran hari ini.'
            ], 409);
        }

        DB::beginTransaction();
        try {
            Kehadiran::create([
                'anggota_id' => $anggota->id,
                'waktu_hadir' => now(),
            ]);

            DB::commit();
            Log::info('Kehadiran berhasil dicatat untuk Anggota ID: ' . $anggota->id . ' (' . $anggota->nama . ')');
            return response()->json([
                'status' => 'success',
                'message' => 'Kehadiran ' . $anggota->nama . ' berhasil dicatat!'
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error recording attendance for UUID ' . $qrCodeData . ': ' . $e->getMessage(), ['exception' => $e]);
            return response()->json([
                'status' => 'error',
                'message' => 'Terjadi kesalahan saat mencatat kehadiran. Silakan coba lagi.'
            ], 500);
        }
    }

    public function showDashboard()
    {
        $today = now()->toDateString();

        // Ambil semua anggota
        $totalAnggota = Anggota::count();

        // Ambil semua kehadiran hari ini
        $kehadiranHariIni = Kehadiran::whereDate('waktu_hadir', $today)->get();
        $hadirIds = $kehadiranHariIni->pluck('anggota_id')->toArray();

        // Hitung anggota yang hadir
        $anggotaHadir = count($hadirIds);

        // Hitung anggota yang tidak hadir
        $anggotaTidakHadir = $totalAnggota - $anggotaHadir;

        // Format chart data untuk statistik
        $chartData = $kehadiranHariIni->groupBy(function ($item) {
            return $item->waktu_hadir->format('Y-m-d'); // Group by date
        })->map(function ($group) {
            return $group->count(); // Count per date
        });

        return Inertia::render('Dashboard', [
            'totalAnggota' => $totalAnggota,
            'anggotaHadir' => $anggotaHadir,
            'anggotaTidakHadir' => $anggotaTidakHadir,
            'chartData' => $chartData->map(function ($count, $date) {
                return ['date' => $date, 'value' => $count];
            })->values(), // Format data for the chart
        ]);
    }
    public function showDaftarKehadiran()
    {
        $today = now()->toDateString();

        // Ambil semua anggota
        $anggotaList = Anggota::all();

        // Ambil semua kehadiran hari ini dan index berdasarkan anggota_id
        $kehadiranHariIni = Kehadiran::whereDate('waktu_hadir', $today)->get()->keyBy('anggota_id');

        // Gabungkan data kehadiran dengan anggota
        $kehadiranFormatted = $anggotaList->map(function ($anggota) use ($kehadiranHariIni) {
            $kehadiran = $kehadiranHariIni->get($anggota->id);

            return [
                'id' => $kehadiran->id ?? null,
                'anggota_nama' => $anggota->nama ?? 'N/A',
                'anggota_no_wa' => $anggota->no_wa ?? 'N/A',
                'anggota_sumber' => $anggota->sumber ?? 'N/A',
                'waktu_hadir' => $kehadiran ? $kehadiran->waktu_hadir->format('H:i:s') : null,
                'tanggal_hadir' => $kehadiran ? $kehadiran->waktu_hadir->format('d-m-Y') : null,
                'status' => $kehadiran ? 'Hadir' : 'Tidak Hadir',
            ];
        });

        return Inertia::render('DaftarKehadiran', [
            'kehadiranData' => $kehadiranFormatted,
            'currentDate' => now()->translatedFormat('l, d F Y'),
        ]);
    }
}
