<?php

namespace App\Http\Controllers;

use App\Models\Anggota;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Storage;
use Ramsey\Uuid\Guid\Guid;
use GuzzleHttp\Client;

class AnggotaController extends Controller
{
    public function index(Request $request) // <--- PASTIKAN ADA Request $request
    {
        $search = $request->input('search');
        $sumber = $request->input('sumber');

        $query = Anggota::query();

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('nama', 'like', '%' . $search . '%')
                    ->orWhere('no_kta', 'like', '%' . $search . '%')
                    ->orWhere('no_wa', 'like', '%' . $search . '%')
                    ->orWhere('email', 'like', '%' . $search . '%')
                    ->orWhere('nama_perusahaan', 'like', '%' . $search . '%')
                    ->orWhere('alamat', 'like', '%' . $search . '%')
                    ->orWhere('sumber', 'like', '%' . $search . '%');
            });
        }

        if ($sumber) {
            $query->where('sumber', $sumber);
        }

        // --- Perubahan Kunci di Sini ---
        $anggotas = $query->paginate(10); // Menggunakan paginate

        return Inertia::render('DataAnggota', [
            'anggotas' => $anggotas->through(fn($anggota) => [ // Menggunakan through untuk transformasi
                'id' => $anggota->id,
                'nama' => $anggota->nama,
                'no_wa' => $anggota->no_wa,
                'email' => $anggota->email,
                'nama_perusahaan' => $anggota->nama_perusahaan,
                'sumber' => $anggota->sumber,
                'wa_sent' => $anggota->wa_sent,
                'email_sent' => $anggota->email_sent,
                'qr_code_path' => $anggota->qr_code_path,
            ]),
            'current_page' => $anggotas->currentPage(),
            'last_page' => $anggotas->lastPage(),
            'links' => $anggotas->linkCollection(),
            'from' => $anggotas->firstItem(),
            'to' => $anggotas->lastItem(),
            'total' => $anggotas->total(),
            'search' => $search,
            'sumber' => $sumber,
        ]);
    }

    public function create()
    {
        // Render halaman form create anggota
        return Inertia::render('anggota.create');
    }

    public function store(Request $request)
    {
        // Validasi input
        $validated = $request->validate([
            'nama' => 'required|string',
            'no_kta' => 'nullable|string',
            'no_wa' => 'required|string',
            'email' => 'required|string',
            'sumber' => 'required|string',
            'nama_perusahaan' => 'required|string',
            'alamat' => 'nullable|string',
        ]);

        // Membuat anggota baru
        $anggota = Anggota::create($validated);
        $anggota->uuid = Guid::uuid4()->toString();
        $anggota->save(); // Pastikan UUID tersimpan

        try {
            // Generate QR Code
            $filename = 'qr_codes/' . $anggota->uuid . '.png';
            $dataToEncode = $anggota->uuid;
            $qrCodeUrl = 'https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=' . urlencode($dataToEncode);

            $client = new Client();
            $response = $client->get($qrCodeUrl, ['verify' => false]);
            $qrImageData = $response->getBody()->getContents();

            if (empty($qrImageData)) {
                throw new \Exception("QR Code image data is empty from API.");
            }

            // Menyimpan QR Code ke disk
            Storage::disk('public')->put($filename, $qrImageData);

            // Menyimpan path QR Code di database
            $anggota->update([
                'qr_code_path' => $filename
            ]);
        } catch (\Exception $e) {
            report($e);
            return redirect()->back()->withErrors(['qr_code_error' => 'Gagal membuat QR Code: ' . $e->getMessage()])->withInput();
        }

        // Redirect ke halaman index
        return redirect()->route('data-anggota.index');
    }

    public function show(Anggota $anggota)
    {
        // Menampilkan detail anggota
        return Inertia::render('DataAnggota/Show', ['anggota' => $anggota]);
    }

    public function edit(Anggota $anggota)
    {
        // Menampilkan form edit anggota
        return Inertia::render('anggota.edit', ['anggota' => $anggota]);
    }

    public function update(Request $request, Anggota $anggota)
    {
        // Validasi input
        $validated = $request->validate([
            'nama' => 'required|string',
            'no_kta' => 'nullable|string',
            'no_wa' => 'required|string',
            'email' => 'required|string',
            'sumber' => 'required|string',
            'nama_perusahaan' => 'required|string',
            'alamat' => 'nullable|string',
        ]);

        // Update data anggota
        $anggota->update($validated);

        // Redirect kembali ke halaman index dengan pesan sukses
        return redirect()->route('data-anggota.index')->with('success', 'Anggota berhasil diupdate!');
    }

    public function destroy(Anggota $anggota)
    {
        // Cek jika QR code ada dan hapus file QR code
        if ($anggota->qr_code_path && Storage::disk('public')->exists($anggota->qr_code_path)) {
            Storage::disk('public')->delete($anggota->qr_code_path);
        }

        // Hapus data anggota
        $anggota->delete();

        // Redirect ke halaman index
        return redirect()->route('data-anggota.index');
    }
}
