<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Anggota;
use Illuminate\Support\Facades\Log;
use App\Jobs\SendWaInvitationJob;
use App\Jobs\SendEmailInvitationJob;
use GuzzleHttp\Client;

class SendWhatsAppInvitation extends Command
{
    protected $signature = 'send:whatsapp-invitation';
    protected $description = 'Mengirimkan undangan Muscab Hipmi melalui WhatsApp dan Email ke semua anggota';

    public function handle()
    {
        // Ambil semua anggota dari database
        $anggotaList = Anggota::all();

        // Base URL untuk QR Code eksternal
        $baseUrlQrCodeEksternal = 'https://bpchipmi.cyberindotech.com/';

        // Loop setiap anggota untuk mengirim undangan
        foreach ($anggotaList as $anggota) {
            $this->info("Processing Anggota ID: {$anggota->id} - {$anggota->nama}");

            // Buat URL QR Code eksternal
            // Kita asumsikan $anggota->qr_code_path berisi nama file QR Code (misal: "anggota_123.png")
            // atau bagian path unik setelah "/qrcodes/"
            $qrCodeUrl = $anggota->qr_code_path ? $baseUrlQrCodeEksternal . $anggota->qr_code_path : null;

            // Persingkat URL jika ada
            $shortUrl = $qrCodeUrl ? $this->shortenUrl($qrCodeUrl) : null;

            // Bangun pesan WhatsApp
            $message = "*[Pesan Otomatis]*\n\n";
            $message .= "Kepada Yth. Bapak/Ibu {$anggota->nama}\n";
            $message .= "Pimpinan dari {$anggota->nama_perusahaan}\n\n";
            $message .= "Dengan hormat,\nKami mengundang Bapak/Ibu untuk hadir dalam acara:\n\n";
            $message .= "*MUSCAB BPC HIPMI Karawang*\n";
            $message .= "🗓 Minggu, 29 Juni 2025\n🕒 09.00 WIB - Selesai\n📍 Hotel Front One Akshaya, Karawang\n\n";
            $message .= "🗾 Unduh QR Code Kepesertaan:\n" . ($shortUrl ?? '(tidak tersedia)') . "\n\n";
            $message .= "Salam hormat,\nPanitia MUSCAB HIPMI";

            // --- Logika untuk WhatsApp ---
            if ($anggota->no_wa) {
                // Cek apakah WA sudah dikirim sebelumnya
                if ($anggota->wa_sent) {
                    $this->warn("    ⏩ WA invitation for {$anggota->nama} (ID: {$anggota->id}) skipped. Already sent.");
                } else {
                    // Dispatch job pengiriman WA jika belum dikirim
                    dispatch(new SendWaInvitationJob($anggota, $message));
                    $this->info("    ✅ WA invitation for {$anggota->nama} (ID: {$anggota->id}) dispatched to queue.");
                }
            } else {
                $this->warn("    ℹ️ No WhatsApp number for {$anggota->nama} (ID: {$anggota->id}). Skipping WA invitation.");
            }

            // --- Logika untuk Email ---
            if ($anggota->email) {
                // **PENTING:** Validasi alamat email sebelum mendispatch job
                if (!filter_var($anggota->email, FILTER_VALIDATE_EMAIL)) {
                    $this->warn("    ⚠️ Invalid email address for {$anggota->nama} (ID: {$anggota->id}): '{$anggota->email}'. Skipping Email invitation.");
                    // Opsional: Perbarui status di database bahwa email tidak valid
                    // Ini akan mencegah job yang sama terus-menerus gagal
                    $anggota->email_sent = false;
                    $anggota->email_error_log = 'Alamat email tidak valid: ' . $anggota->email;
                    $anggota->save();
                    continue; // Lanjut ke anggota berikutnya
                }

                // Cek apakah email sudah dikirim sebelumnya
                if ($anggota->email_sent) {
                    $this->warn("    ⏩ Email invitation for {$anggota->nama} (ID: {$anggota->id}) skipped. Already sent.");
                } else {
                    // Dispatch job pengiriman email jika belum dikirim dan email valid
                    dispatch(new SendEmailInvitationJob($anggota, $shortUrl));
                    $this->info("    ✅ Email invitation for {$anggota->nama} (ID: {$anggota->id}) dispatched to queue.");
                }
            } else {
                $this->warn("    ℹ️ No email address for {$anggota->nama} (ID: {$anggota->id}). Skipping Email invitation.");
            }
        }

        $this->info('---');
        $this->info('✅ Semua undangan telah diproses dan didispatch ke queue (jika belum terkirim).');
    }

    /**
     * Mempersingkat URL menggunakan TinyURL API.
     *
     * @param string $url URL asli yang akan dipersingkat.
     * @return string URL yang sudah dipersingkat atau URL asli jika gagal.
     */
    public function shortenUrl($url)
    {
        try {
            $client = new Client();
            $response = $client->get("https://tinyurl.com/api-create.php?url=" . urlencode($url));
            return (string) $response->getBody();
        } catch (\Exception $e) {
            Log::error('Short URL Error: ' . $e->getMessage());
            return $url; // Kembalikan URL asli jika gagal mempersingkat
        }
    }
}
